//
//
// USB control code for Analog_testboard FPGA
//
// jb/march/2019 version v1.0
//
//
module USB_main_control
     (  
        input clk40, 
        //I2C interface (1.2V) 
        output SCL,      //I2C clock  ...no pullup resistor on this line
        input  SDA_i,   
        output RSTB,   
        //output wire[1:0] chip_id,      //fixed I2C address inside this version of COLUTA 
        output sda_padoen_oe,
		  output sda_pad_o,
	
        //Bunch crossing reset signal
        output BCRST,

		  
        // USB interface signals
        inout wire [7:0] DATA,  //fifo DATA
        input       RXF,        //low active: data available in FIFO
        input       TXE,        //low active: data can be written into FIFO
        output      RD,         //ReaD enable,  low to high transition
        output      WR,         //WRite enable, high to low transition

		  
        //programmable delay trigger    
        output reg trigger,      //interface to external pulser   (3.3V signaling)
		  
		  		  
        // test points
        output wire TP1,
        output wire TP2,
        output wire TP_dummy    //just to remove warnings about signals which do not drive logic
     );
	
	
//wires and registers
wire   [7:0] I2C_rdback_mux, misc_mux;
reg  [639:0] I2C_cntrl;
wire [639:0] I2C_shift_back;
wire         clk5_int;
reg    [3:0] stw_cnt_rd, stw_cnt_wr;
reg          st_byte_read, st_byte_write;
reg          eoc_rd;
reg    [7:0] status1, status2, status3, status4, status5, status6;
reg   [13:0] BC_creg, BC_reg;
reg          fA_write, eoc_wr_A; 

reg   [63:0] ad_reg, oc_reg; 
reg          c_reg_load;

reg   [16:0] fAw_cnt_rd;
reg    [6:0] fAw_cnt_wr;
reg          fA_read, eoc_rd_A;
reg          status_rd_pulse, status_read_op;
reg          A_start_del, FIFO_A_op_start, FIFO_A_operation;
reg          rst, RST_N_flag;
reg    [4:0] rst_flg;
reg   [13:0] adc_counter_320;
reg          adc_write_op, adc_write_op_sync, eof_ADC_write;


				 
reg          start_op, dp_write_320, dp_write_320_del;
wire         clk320_trigger;
wire         last_i2c_ack, I2C_USB_done; 
wire         clk40_int, pll_40M_locked;

wire         scl_padoen_oe;
//
// 	Code body starts here
//

	

	pll_40MHz p0 
	(
		.rst      (rst),            // reset.reset
		.refclk   (clk40),          // refclk.clk
		.locked   (pll_40M_locked), // locked.export
		.outclk_0 (clk5_int),        
		.outclk_1 (clk40_int),       
		.outclk_2 (clk320_trigger)   
	);


		  
//*******************************************************************************
//******************************FIFO B part**************************************
//*******************************************************************************
//FIFO B is used for status register read/write operation
//this register controls all operations in the FPGA
//
//Status word description
//byte status1:
//   b[7:0] "11111110" (FE, read only)
//
//byte status2:
//   b[1:0] fifo A operation type flags:
//
//          2'b00 no-op
//          2'b01 FPGA control register write;
//          2'b10 FPGA control register read;
//          2'b11 Read memory data
//
//   b[4:2] control/data register address
//          3'b000 I2C    
//          3'b001 
//          3'b010 DAC    
//          3'b011 
//          3'b100 histogram control
//          3'b101 BC
//          3'b110 Coluta
//
//   
// 
//   b[5]    resetB for COLUTA chip
//   b[6]    reserved
//   b[7]    reserved 
//
//byte status3:
//   b[7:0]  fifo A read/write byte counter (low  byte)
//           For control data write operation it (5 least signficant bits are used)
//           defines the number of bytes sent to internal buffer.
//           
//
//byte status4:
//   b[7:0]  fifo A read byte counter (high byte)
//           
//byte status5:                 
//   b[0]    start control data operation (execute data register commands)
//   b[1]    pulse command (increment pulse delay counter)  
//   b[2]    COLUTA analog measurement start
//   b[3]    reset trigger pulse delay
//
//   b[4]    reserved
//   b[5]    software reset (0-1 transition starts operation)
//   b[6]    start FIFO A operation, load FIFO A data into FPGA control register (0-1 transition starts operation)
//                                or read FPGA control register to FIFO A
//   b[7]    read the status word (0-1 transition starts operation)
//
//
//byte status6:
//   b[7:0]  all "11111111" (FF, read only)
//*******************************************************************************


//
//status word write operation. Status data are received from fifo B 
//
reg [3:0] rst_del;
always @ (posedge clk5_int)
   begin

        //status write operation is always executed as soon as there are data
        //in the fifo B output
        //PC is the master of status comunication, it must deliver exactly 6 bytes of status !!!!!!!!!!!!!!
        if      (RXF==1'b1)            st_byte_write <= 1'b0;
        else if (st_byte_write==1'b1)  st_byte_write <= 1'b0;
        else if (RXF==1'b0)            st_byte_write <= 1'b1;
 
        //counter for number of status bytes to write
        if      ((st_byte_write==1'b1) & (DATA[7:0]==8'b11111111)) stw_cnt_wr <= 4'b0000;
        else if ((st_byte_write==1'b1) & (DATA[7:0]==8'b11111110)) stw_cnt_wr <= 4'b0001;
        else if ( st_byte_write==1'b1)                             stw_cnt_wr <= stw_cnt_wr + 4'b0001;
		
        //reset
        rst_flg[4:0] <= {rst_flg[3:0],status5[5]};   //status word reset
        rst_del[0]   <= (rst_flg[3] & !rst_flg[4]);
        rst_del[3:1] <= rst_del[2:0];
        rst          <= rst_del[3];
		
        //status registers
        if     (st_byte_write==1'b1 & stw_cnt_wr==4'b0000) status1[7:0] <= DATA[7:0];
        else if(st_byte_write==1'b1 & stw_cnt_wr==4'b0001) status2[7:0] <= DATA[7:0];
        else if(st_byte_write==1'b1 & stw_cnt_wr==4'b0010) status3[7:0] <= DATA[7:0];
        else if(st_byte_write==1'b1 & stw_cnt_wr==4'b0011) status4[7:0] <= DATA[7:0];
        else if(st_byte_write==1'b1 & stw_cnt_wr==4'b0100) status5[7:0] <= DATA[7:0];
        else if(st_byte_write==1'b1 & stw_cnt_wr==4'b0101) status6[7:0] <= DATA[7:0];        
   end

//this is a read operation for FT2232!!!!	
      assign RD = !(st_byte_write) | !(fA_write);  

//
//status word read operation. The data are sent to fifo B 
//
reg [5:0] stat_del;
always @ (posedge clk5_int)
   begin

        //leading edge of status5[7] starts read operation
        //restart is posible by writing into status5[7] sequence 0 1 
        stat_del[0]     <= status5[7];
        stat_del[5:1]   <= stat_del[4:0];
        status_rd_pulse <= !stat_del[5] & status5[7];
		
        if (rst==1'b1 | eoc_rd==1'b1)   status_read_op <= 1'b0;
        else if (status_rd_pulse==1'b1) status_read_op <= 1'b1;
		
        //counter for number of status bytes to be read
        if (rst==1'b1 | eoc_rd==1'b1) stw_cnt_rd <= 4'b0000;
        else if (st_byte_read==1'b1)  stw_cnt_rd <= stw_cnt_rd + 4'b0001;
	
        //status read operation is initiated by status word write operation
        //by setting    status5[7] (read the status word flag) to 1 
        if      (rst==1'b1 | TXE==1'b1)  st_byte_read <= 1'b0;
        else if (st_byte_write==1'b1)    st_byte_read <= 1'b0;
        else if (st_byte_read==1'b1)     st_byte_read <= 1'b0;
        //read status is possible only after write operation is finished
        else if (TXE==1'b0 & status_read_op & stw_cnt_wr==4'b0000)  
                                         st_byte_read <= 1'b1;
	
        //end of read cycle
        if (st_byte_read & (stw_cnt_rd==4'b0101)) eoc_rd <= 1'b1;
        else                                      eoc_rd <= 1'b0;
		
	end

	
//be carefull, this is a write operation for FT2232 !!!
	assign WR = st_byte_read | fA_read;	

//mux for status register read
	assign DATA   = (st_byte_read==1'b1 & stw_cnt_rd==4'b0000) ? status1         :
	                (st_byte_read==1'b1 & stw_cnt_rd==4'b0001) ? status2         :
	                (st_byte_read==1'b1 & stw_cnt_rd==4'b0010) ? status3         :
	                (st_byte_read==1'b1 & stw_cnt_rd==4'b0011) ? status4         :
	                (st_byte_read==1'b1 & stw_cnt_rd==4'b0100) ? status5         :	
	                (st_byte_read==1'b1 & stw_cnt_rd==4'b0101) ? status6         :
                   (fA_read==1'b1 & status2[4:0]==5'b00010)   ? I2C_rdback_mux  :  
                   (fA_read==1'b1 & status2[4:0]==5'b01110)   ? misc_mux        :  8'bZZZZZZZZ;				
		
//		
//*******************************************************************************
//******************************FIFO A part**************************************
//*******************************************************************************
// fifo A is used to read/write data from/to COLUTA chip
//

//
//send data from fifo A into FPGA 
//
always @ (posedge clk5_int)
   begin

        //leading edge of status5[6] starts FIFO A operation
        A_start_del     <= status5[6];
        FIFO_A_op_start <= !A_start_del & status5[6];
		
        if (rst==1'b1 | eoc_wr_A==1'b1 | eoc_rd_A==1'b1)  FIFO_A_operation <= 1'b0;
        else if (FIFO_A_op_start==1'b1)                 FIFO_A_operation <= 1'b1;
		
        //counter for number of data bytes to write
        if (rst==1'b1 | eoc_wr_A==1'b1) fAw_cnt_wr <= 7'b0000001;
        else if (fA_write==1'b1)        fAw_cnt_wr <= fAw_cnt_wr + 7'b0000001;
	
        //data byte write operation is initiated by PC software
        if (rst==1'b1 | RXF==1'b1 | status2[1:0]!=2'b01) fA_write <= 1'b0;
        else if (fA_write==1'b1)                         fA_write <= 1'b0;
        else if (RXF==1'b0 & FIFO_A_operation==1'b1)     fA_write <= 1'b1;
	
        //end of write cycle
        if (FIFO_A_operation==1'b0)                                     eoc_wr_A <= 1'b0;
        else if ((fAw_cnt_wr[6:0]==status3[6:0])&(status2[1:0]==2'b01)) eoc_wr_A <= 1'b1;
        else                                                            eoc_wr_A <= 1'b0;	
		
		  //64 bit scan chain control register
		  if (FIFO_A_op_start==1'b1)                          I2C_cntrl <= {640{1'b0}};
		  else if ((status2[4:0]==5'b00001)&(fA_write==1'b1)) I2C_cntrl <= { DATA, I2C_cntrl[639:8] };
		 

		  //14 bit BC control register
		  if (FIFO_A_op_start==1'b1)                          BC_reg <= {14{1'b0}};
		  else if ((status2[4:0]==5'b10101)&(fA_write==1'b1)) BC_reg <= { DATA, BC_reg[13:8] };			  
		  
		  c_reg_load <= eoc_wr_A;
		  			 
		  if ((c_reg_load==1'b1)&(status2[4:0]==5'b10101)) BC_creg     <= BC_reg;  
	end

	
//
//send data from FPGA into fifo A 
//
reg fA_read_del;
always @ (posedge clk5_int)
   begin
	
    //counter for number of status bytes to write
    //before you start read operation send 2'b00 to status2[1:0]
    if (rst==1'b1 | eoc_rd_A==1'b1 | (status2[1:0]==2'b00 | status2[1:0]==2'b01)) 
                                     fAw_cnt_rd <= 17'b00000000000000000;
    else if (fA_read==1'b1)          fAw_cnt_rd <= fAw_cnt_rd + 17'b00000000000000001;
	
    //FPGA data read operation is active
    //as soon as status register allows it
    if (rst==1'b1 | status2[1:0]==2'b00 | status2[1:0]==2'b01)     fA_read <= 1'b0;
    else if (fA_read==1'b1)                                        fA_read <= 1'b0;
    else if (fA_read_del==1'b1)                                    fA_read <= 1'b0;
    else if (TXE==1'b1)                                            fA_read <= 1'b0;
    else if (TXE==1'b0 & FIFO_A_operation==1'b1 & eoc_rd_A== 1'b0) fA_read <= 1'b1;
	
    fA_read_del <= fA_read; // To provide delay to the counter so as to avoid extra clock cycle after EOC for reg  A is issued.
		
    //end of read cycle is set to "1" if
    //number of bytes transmitted agrees with status word counter
    if ((fAw_cnt_rd=={status4[4:0],status3,4'b0000})&      // 16bytes per sample
        (status2[1:0]==2'b10 | status2[1:0]==2'b11)) eoc_rd_A <= 1'b1;
    else                                             eoc_rd_A <= 1'b0;
		
   end



	
				 
	
	
 //
 // mux for control register read (to fifo A)
 //

	assign misc_mux =    (fAw_cnt_rd[2:0]==3'b000) ? {1'b0,   pll_40M_locked, I2C_USB_done, last_i2c_ack, 4'b0000} : 8'b00000000;							
													
	
 //
 // mux for I2C shift register read (to fifo A)
 //
   assign I2C_rdback_mux = 
                       (fAw_cnt_rd[6:0]==7'b0000000) ? I2C_shift_back[  7:   0] :
	                    (fAw_cnt_rd[6:0]==7'b0000001) ? I2C_shift_back[ 15:   8] :
	                    (fAw_cnt_rd[6:0]==7'b0000010) ? I2C_shift_back[ 23:  16] :	
	                    (fAw_cnt_rd[6:0]==7'b0000011) ? I2C_shift_back[ 31:  24] :
	                    (fAw_cnt_rd[6:0]==7'b0000100) ? I2C_shift_back[ 39:  32] :
	                    (fAw_cnt_rd[6:0]==7'b0000101) ? I2C_shift_back[ 47:  40] : 
	                    (fAw_cnt_rd[6:0]==7'b0000110) ? I2C_shift_back[ 55:  48] : 
	                    (fAw_cnt_rd[6:0]==7'b0000111) ? I2C_shift_back[ 63:  56] : 
	                    (fAw_cnt_rd[6:0]==7'b0001000) ? I2C_shift_back[ 71:  64] : 							 
	                    (fAw_cnt_rd[6:0]==7'b0001001) ? I2C_shift_back[ 79:  72] : 
	                    (fAw_cnt_rd[6:0]==7'b0001010) ? I2C_shift_back[ 87:  80] : 
	                    (fAw_cnt_rd[6:0]==7'b0001011) ? I2C_shift_back[ 95:  88] : 
                       (fAw_cnt_rd[6:0]==7'b0001100) ? I2C_shift_back[103:  96] : 
                       (fAw_cnt_rd[6:0]==7'b0001101) ? I2C_shift_back[111: 104] : 
                       (fAw_cnt_rd[6:0]==7'b0001110) ? I2C_shift_back[119: 112] :
                       (fAw_cnt_rd[6:0]==7'b0001111) ? I2C_shift_back[127: 120] :
							 
                       (fAw_cnt_rd[6:0]==7'b0010000) ? I2C_shift_back[135: 128] :
	                    (fAw_cnt_rd[6:0]==7'b0010001) ? I2C_shift_back[143: 136] :
	                    (fAw_cnt_rd[6:0]==7'b0010010) ? I2C_shift_back[151: 144] :	
	                    (fAw_cnt_rd[6:0]==7'b0010011) ? I2C_shift_back[159: 152] :
                       (fAw_cnt_rd[6:0]==7'b0010100) ? I2C_shift_back[167: 160] :
                       (fAw_cnt_rd[6:0]==7'b0010101) ? I2C_shift_back[175: 168] : 
                       (fAw_cnt_rd[6:0]==7'b0010110) ? I2C_shift_back[183: 176] : 
                       (fAw_cnt_rd[6:0]==7'b0010111) ? I2C_shift_back[191: 184] : 
                       (fAw_cnt_rd[6:0]==7'b0011000) ? I2C_shift_back[199: 192] : 							 
                       (fAw_cnt_rd[6:0]==7'b0011001) ? I2C_shift_back[207: 200] : 
                       (fAw_cnt_rd[6:0]==7'b0011010) ? I2C_shift_back[215: 208] : 
                       (fAw_cnt_rd[6:0]==7'b0011011) ? I2C_shift_back[223: 216] : 
                       (fAw_cnt_rd[6:0]==7'b0011100) ? I2C_shift_back[231: 224] : 
                       (fAw_cnt_rd[6:0]==7'b0011101) ? I2C_shift_back[239: 232] : 
                       (fAw_cnt_rd[6:0]==7'b0011110) ? I2C_shift_back[247: 240] :
                       (fAw_cnt_rd[6:0]==7'b0011111) ? I2C_shift_back[255: 248] :
							 
                       (fAw_cnt_rd[6:0]==7'b0100000) ? I2C_shift_back[263: 256] :
	                    (fAw_cnt_rd[6:0]==7'b0100001) ? I2C_shift_back[271: 264] :
	                    (fAw_cnt_rd[6:0]==7'b0100010) ? I2C_shift_back[279: 272] :	
	                    (fAw_cnt_rd[6:0]==7'b0100011) ? I2C_shift_back[287: 280] :
                       (fAw_cnt_rd[6:0]==7'b0100100) ? I2C_shift_back[295: 288] :
                       (fAw_cnt_rd[6:0]==7'b0100101) ? I2C_shift_back[303: 296] : 
                       (fAw_cnt_rd[6:0]==7'b0100110) ? I2C_shift_back[311: 304] : 
                       (fAw_cnt_rd[6:0]==7'b0100111) ? I2C_shift_back[319: 312] : 
                       (fAw_cnt_rd[6:0]==7'b0101000) ? I2C_shift_back[327: 320] : 							 
                       (fAw_cnt_rd[6:0]==7'b0101001) ? I2C_shift_back[335: 328] : 
                       (fAw_cnt_rd[6:0]==7'b0101010) ? I2C_shift_back[343: 336] : 
                       (fAw_cnt_rd[6:0]==7'b0101011) ? I2C_shift_back[351: 344] : 
                       (fAw_cnt_rd[6:0]==7'b0101100) ? I2C_shift_back[359: 352] : 
                       (fAw_cnt_rd[6:0]==7'b0101101) ? I2C_shift_back[367: 360] : 
                       (fAw_cnt_rd[6:0]==7'b0101110) ? I2C_shift_back[375: 368] :
                       (fAw_cnt_rd[6:0]==7'b0101111) ? I2C_shift_back[383: 376] :
							 
                       (fAw_cnt_rd[6:0]==7'b0110000) ? I2C_shift_back[391: 384] :
	                    (fAw_cnt_rd[6:0]==7'b0110001) ? I2C_shift_back[399: 392] :
	                    (fAw_cnt_rd[6:0]==7'b0110010) ? I2C_shift_back[407: 400] :	
	                    (fAw_cnt_rd[6:0]==7'b0110011) ? I2C_shift_back[415: 408] :
                       (fAw_cnt_rd[6:0]==7'b0110100) ? I2C_shift_back[423: 416] :
                       (fAw_cnt_rd[6:0]==7'b0110101) ? I2C_shift_back[431: 424] : 
                       (fAw_cnt_rd[6:0]==7'b0110110) ? I2C_shift_back[439: 432] : 
                       (fAw_cnt_rd[6:0]==7'b0110111) ? I2C_shift_back[447: 440] : 
                       (fAw_cnt_rd[6:0]==7'b0111000) ? I2C_shift_back[455: 448] : 							 
                       (fAw_cnt_rd[6:0]==7'b0111001) ? I2C_shift_back[463: 456] : 
                       (fAw_cnt_rd[6:0]==7'b0111010) ? I2C_shift_back[471: 464] : 
                       (fAw_cnt_rd[6:0]==7'b0111011) ? I2C_shift_back[479: 472] : 
                       (fAw_cnt_rd[6:0]==7'b0111100) ? I2C_shift_back[487: 480] : 
                       (fAw_cnt_rd[6:0]==7'b0111101) ? I2C_shift_back[495: 488] : 
                       (fAw_cnt_rd[6:0]==7'b0111110) ? I2C_shift_back[503: 496] :
                       (fAw_cnt_rd[6:0]==7'b0111111) ? I2C_shift_back[511: 504] :
							 
                       (fAw_cnt_rd[6:0]==7'b1000000) ? I2C_shift_back[519: 512] :
	                    (fAw_cnt_rd[6:0]==7'b1000001) ? I2C_shift_back[527: 520] :
	                    (fAw_cnt_rd[6:0]==7'b1000010) ? I2C_shift_back[535: 528] :	
	                    (fAw_cnt_rd[6:0]==7'b1000011) ? I2C_shift_back[543: 536] :
                       (fAw_cnt_rd[6:0]==7'b1000100) ? I2C_shift_back[551: 544] :
                       (fAw_cnt_rd[6:0]==7'b1000101) ? I2C_shift_back[559: 552] : 
                       (fAw_cnt_rd[6:0]==7'b1000110) ? I2C_shift_back[567: 560] : 
                       (fAw_cnt_rd[6:0]==7'b1000111) ? I2C_shift_back[575: 568] : 
                       (fAw_cnt_rd[6:0]==7'b1001000) ? I2C_shift_back[583: 576] : 							 
                       (fAw_cnt_rd[6:0]==7'b1001001) ? I2C_shift_back[591: 584] : 
                       (fAw_cnt_rd[6:0]==7'b1001010) ? I2C_shift_back[599: 592] : 
                       (fAw_cnt_rd[6:0]==7'b1001011) ? I2C_shift_back[607: 600] : 
                       (fAw_cnt_rd[6:0]==7'b1001100) ? I2C_shift_back[615: 608] : 
                       (fAw_cnt_rd[6:0]==7'b1001101) ? I2C_shift_back[623: 616] : 
                       (fAw_cnt_rd[6:0]==7'b1001110) ? I2C_shift_back[631: 624] :
                       (fAw_cnt_rd[6:0]==7'b1001111) ? I2C_shift_back[639: 632] :8'b00000000;	 

//*******************************************************************************							 
//*******************************************************************************
//reset and  start flags (common synchronization)
//*******************************************************************************							 
//*******************************************************************************

reg  reset_1, BC_reset, 
     i2c_reset,
     analog_start1, analog_start, pulser_start,
     control_start1, usb_i2c_start,
     usb_BC_start;

reg [4:0] control_start_del, analog_start_del;
	  
always @(posedge clk40_int) 
  begin	
   reset_1        <= rst;
   
//leading edge detected	 
   control_start_del  <= (control_start_del << 1) | status5[0];
   analog_start_del   <= (analog_start_del  << 1) | status5[2];   
   
   control_start1 <= ~control_start_del[4] & control_start_del[3];
   analog_start1  <= ~analog_start_del [4] & analog_start_del [3];
  end
   
always @(posedge clk40_int) 	
  begin		
	
   i2c_reset    <= reset_1;
   BC_reset <= reset_1;

	 
   analog_start <= analog_start1;
   pulser_start <= analog_start1;	 
	 
   usb_i2c_start <= (status2[4:0]==5'b00001) & control_start1;
   usb_BC_start  <= (status2[4:0]==5'b10101) & control_start1;
	 
  end
//*******************************************************************************							 
//*******************************************************************************



//****************************************************************
//                   ***** I2C state machine *****
//****************************************************************
// procedure:
//       a/PC software loads control data register through fifo A
//         (USB control data stream)
//       b/PC software starts the execution of the I2C instruction
//         in the control data stream
//       c/I2C state machine(below) automaticaly ecexutes 
//         all I2C commands in the queue
//
//
wire [7:0] wb_dat_o,wb_dat_i;
wire [2:0] wb_adr_i;
wire       wb_ack_o, wb_we_i, wb_stb_i, wb_cyc_i, 
           scl_pad_o;
wire       scl_pad_i, sda_pad_i;	


		  
//start flag
reg[2:0] I2C_start;
always @(posedge clk40_int)
begin
	I2C_start[1:0] <= {I2C_start[0],  (usb_i2c_start)};
	I2C_start[2]   <= ~I2C_start[1] & I2C_start[0];
end


 

//
//state machine converts USB control data stream into control sequence
//for Wishbone compatible I2C master core
// 

 USB_I2C_state_machine usb_i2c_sm
(
	// 
	.clk                (clk40_int),
	.rst                (i2c_reset),
	
	// I2C master core signals
	.i2c_reg_adr        (wb_adr_i),
	.data_to_i2c        (wb_dat_i),
	.data_from_i2c      (wb_dat_o),
	.i2c_we             (wb_we_i), 
	.i2c_stb            (wb_stb_i), 
	.i2c_cyc            (wb_cyc_i), 
	.wb_ack_o           (wb_ack_o),

	//USB command stream
	.i2c_operation_start(I2C_start[2]),
	.USB_cntr_in        (I2C_cntrl),
	.USB_data_out       (I2C_shift_back),
	.i2c_queue_done     (I2C_USB_done),
	.last_i2c_ack       (last_i2c_ack), 
	.usb_tst1           ()             //test only
	);
 
				 
//I2C interface to NevisADC chip
i2c_master_top    I2C
                      (
	                   .wb_clk_i (clk40_int),      // master clock input
							 .wb_rst_i (1'b0),       // synchronous active high reset
							 .arst_i   (~i2c_reset), // asynchronous reset
							 .wb_adr_i (wb_adr_i),   // lower address bits (3bits)
							 .wb_dat_i (wb_dat_i),   // databus input      (8bits)
							 .wb_dat_o (wb_dat_o),   // databus output     (8bits)
							 .wb_we_i  (wb_we_i),    // write enable input
							 .wb_stb_i (wb_stb_i),   // stobe/core select signal
							 .wb_cyc_i (wb_cyc_i),   // valid bus cycle input
							 .wb_ack_o (wb_ack_o),   // bus cycle acknowledge output
							 .wb_inta_o (),          // interrupt request signal output
							 
							 .scl_pad_i (scl_pad_i), //I2C bus signals
							 .scl_pad_o (), 
							 .scl_padoen_o (scl_padoen_oe), 
							 .sda_pad_i (sda_pad_i), 
							 .sda_pad_o (sda_pad_o), 
							 .sda_padoen_o (sda_padoen_oe) 
							 ); 

						

//I2C bus signals
//	assign SCL       = scl_padoen_oe ? 1'bZ : scl_pad_o;
//solution with no pull up resistor on SCL 
   assign SCL       = scl_padoen_oe;
//	assign SDA       = sda_padoen_oe ? 1'bZ : sda_pad_o;
	assign scl_pad_i = SCL;
	assign sda_pad_i = SDA_i;
				 
	assign RSTB         = !status2[5];       //resetB  for COLUTA I2C&chip
//chip_id is set inside COLUTA  to "001"
//assign chip_id[2:0] = {3'b001};           //chip id for COLUTA I2C protocol	





//***********************************************************************		
//***********************Bunch crossing signal***************************
//***********************************************************************
//
//this is to start BC sequence
//
reg[2:0] BC_start;
always @(posedge clk40_int)
begin
	BC_start[1:0] <= {BC_start[0],  (usb_BC_start)};
	BC_start[2]   <= ~BC_start[1] & BC_start[0];
	end 
 
 
 //sequencer
 BC_control BC_1
     (  
        //module state machine inputs
        .clk         (clk40_int),
		  .rst         (BC_reset),
		  .op_start    (BC_start[2]),
		  .BC_cntrl(BC_creg),
		  
		  
		  //BC signal
		  .BCRST       (BCRST)

		);
 


//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!	 
//*******************************************************************************
//***********external pulser interface - programmable delay trigger**************
//*******************************************************************************
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


reg [3:0] delay_reg, delay_counter;
reg [9:0] delay_start;
reg [45:0]external_pulser_start;
reg       pulse_command, trigger_operation;

always @ (posedge clk40_int)
   begin      
        external_pulser_start[0]    <= pulser_start;
        external_pulser_start[45:1] <= external_pulser_start[44:0];                      //COLUTA analog measurement start
        
        if(status5[3])                 delay_reg[3:0] <= 4'b1001;                        //reset the trigger delay counter
        else if (!external_pulser_start[35] && external_pulser_start[34] && status5[1])
        begin
                                       delay_reg <= delay_reg - 4'b0001;                 //decrement delay counter
        end
   end	  

always @ (posedge clk320_trigger)
   begin 
        delay_start[0]   <= external_pulser_start[2];
        delay_start[9:1] <= delay_start[8:0];
 
        if (delay_start[9])                   trigger_operation <= 1'b1;
		  else if (delay_counter[3:0]==4'b0000) trigger_operation <= 1'b0;
                 
        if(!delay_start[3] && delay_start[2]) delay_counter <= delay_reg;        
        else if(trigger_operation)            delay_counter <= delay_counter - 4'b0001;
      
        
       if (external_pulser_start[45])         pulse_command <= 0; 
       else if(delay_counter[3:0]==4'b0000)   pulse_command <= 1; 
		 
		 trigger <= pulse_command;
   end

 
//    
//test points  
//these test points have the PCB routing
//
   assign TP1 =   clk5_int;    
   assign TP2 =   clk40_int; 

//	Dummy test point
//	just to remove warning about signals which do not drive logic
// assigned pin position is T17, with io standard 3.3V CMOS
//
   assign TP_dummy  = analog_start;
	
	endmodule
